# FlexiRPG -- library of miniature templates.
#
# Copyright (C) 2010 David Vrabel
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
import os
from string import *
import orpg.dirpath
from orpg.orpgCore import open_rpg

class MiniatureTemplate(object):
    def __init__(self, url, name=None):
        """Create a miniature template.

        url: URL for the miniature image.
        name: Name for the miniature (optional).

        If no name is specified a name will be generated from the file
        name in the URL.
        """
        if not name:
            name = rsplit(url, '/', 1)[-1]
            name = os.path.splitext(name)[0]
            name = replace(name, '_', ' ')
        self._url = url
        self._name = name
        self._serial = 0

    def __get_url(self):
        return self._url

    def __get_name(self):
        return self._name

    url = property(__get_url)
    name = property(__get_name)

    def new_label(self):
        self._serial += 1
        return "%s %d" % (self.name, self._serial)

class MiniatureLib(object):
    def __init__(self):
        """Create a miniature template library.

        The library is populated from the configuration file
        ~/.flexirpg/miniatures.xml
        """
        self.validate = open_rpg.get_component('validate')
        self.xml = open_rpg.get_component('xml')

        self._library = []

        self.validate.config_file("miniatures.xml", "default_miniatures.xml")
        self.load_from_xml(orpg.dirpath.dir_struct["user"] + "miniatures.xml")

    def load_from_xml(self, filename):
        try:
            f = open(filename, "r")
            xml_doc = self.xml.parseXml(f.read())
            f.close()
        except:
            self.log.log("Warning: Miniature library could not be loaded", ORPG_WARN, True)
            return

        for t in xml_doc.documentElement.getElementsByTagName("miniature"):
            url = None
            name = None
            if t.hasAttribute("url"):
                url = t.getAttribute("url")
            if t.hasAttribute("name"):
                name = t.getAttribute("name")
            if url:
                self._library.append(MiniatureTemplate(url, name))

    def __iter__(self):
        for i in self._library:
            yield i
